﻿// ---------------------
//   Author:  O.Feucht 
// ---------------------

using UnityEngine;
using UnityEngine.Audio;
using System.Collections;

public class MusicController : MonoBehaviour {

	// Zustaände, die der AudioController annehmen kann
	enum MusicController_State{
		mute,	// Keine Musik wird abgespielt, bis AudioController von extern wieder aktiviert wird
		fadeIn,
		fadeOut,
		play,
	}

	// Aktueller Zustand des AudioControllers
	MusicController_State state;

	// Audio Sorce, wo Musik abgespielt werden soll
	public AudioSource audioSource = new AudioSource();

	// Wert bis zu dem Musik beim FadeOut verringert wird
	public float faidOutMax = 0f;

	// Wert bis zu dem Musik beim fadeIn erhöht wird
	public float faidInMax = 1f;

	// Gibt an, um wie viel die Musik beim Fade-In pro Sekunde lauter wird 
	public float volIncPerSec = 0.2f;

	// Gibt an, um wie viel die Musik beim Fade-Out pro Sekunde leiser wird 
	public float volDecPerSec = 0.2f;



	// Gibt an ob nach Ende eines Musikstücks das nächste Anfangen soll
	public bool loop;

	// Hilfsvariable, die angibt ob nach einer AKtion ein weiteres Lied gespielt werden soll
	private bool playNextSong = false;
	
	//Array mit allen Songs für die Audioquelle
	public AudioClip[] musicList;

	// Initialisiert MusikController
	void Start() {
		// Initalisiere Zufallsgenarator
		Random.seed = System.DateTime.Now.GetHashCode();
	}

	/// <summary>
	/// Stoppt das abspielen der Musik
	/// </summary>
	public void StoppMusik(bool fadeOut = true) {
		if (fadeOut == true) {
			playNextSong = false;
			state = MusicController_State.fadeOut;
		} else {
			audioSource.Stop();
			state = MusicController_State.mute;
		}

	}

	/// <summary>
	/// Legt neue Abspielliste fest
	/// </summary>
	/// <param name="newMusicList">New music list.</param>
	public void SetNewMusicList(AudioClip[] newMusicList) {
		this.musicList = newMusicList;
	}

	/// <summary>
	/// Startet ein zufällige Musikstück, von der Liste
	/// </summary>
	/// <param name="loop">If set to <c>true</c> loop.</param>
	public void StartRandomClip(bool loop = true) {
		int rndNumber = Mathf.RoundToInt(Random.Range (0, musicList.Length)) % musicList.Length;
		StartNewClip(musicList [rndNumber], loop);
	}

	/// <summary>
	/// Startet ein neues Musikstück
	/// </summary>
	public void StartNewClip(AudioClip clip, bool loop = true) {
		this.loop = loop;

		// Falls bereits ein Lied abgespielt wird, beende dieses zuerst (durch Fade-Out)
		if (audioSource.isPlaying == true) {
			playNextSong = true;
			state = MusicController_State.fadeOut;
		} else {
			playNextSong = false;
			audioSource.clip = clip;
			audioSource.Play ();
			audioSource.volume = 0.0f;
			state = MusicController_State.fadeIn;
		}
	}

	// Wird jeden Frame aufgerufen
	void Update() {

		switch (state) {

			// Musik ist ausgeschaltet -> Leerlauf
		case MusicController_State.mute:
			break;

			// Lasse Musickstück imme rlauter werden
		case MusicController_State.fadeIn:
			if(audioSource.volume < faidInMax) {
				audioSource.volume += volIncPerSec * Time.deltaTime;
			}else{
				state = MusicController_State.play;
			}
			break;

			// Lasse Musik langsam leiser werden und spiele dann neues Musikstück ab
		case MusicController_State.fadeOut:
			if(audioSource.volume > faidOutMax) {
				audioSource.volume -= volDecPerSec * Time.deltaTime;
			}else{
				if(playNextSong == true) {
					StartRandomClip();
				}else{
					state = MusicController_State.mute;
				}
			}
			break;

			// Spiele Musikstück weiter ab, falls noch nicht am Ende.
			// Ansonsten starte neues Musikstück
		case MusicController_State.play:
			// Warte bis Musickstück vorbei
			if(audioSource.isPlaying ==  false) {
				// Starte neue Musik, falls "loop" aktiviert ist
				if(loop == true) {
					StartRandomClip();
				}
			}
			break;

		}
	}


}
