﻿// ---------------------
//   Author:  O.Feucht 
// ---------------------

using UnityEngine;
using UnityEngine.UI;
using System.Collections;

public class BatteryController : MonoBehaviour {

	public float startingBattery = 100;                          // The amount of battery the player starts the game with.
	private float currentBattery;                                 // The current battery the player has.
	public Slider batterySlider;                                 // Reference to the UI's battery bar.

	// Geschwindigkeit, mit der sich Batterie pro Sekunde verändert
	private float batChangePerSec = 2;								 

	// Aktueller Zustand
	private BatteryControllerState curState;

	// Initialisiert Script
	void Start ()
	{
		curState = BatteryControllerState.IDLE;
		currentBattery = startingBattery;
	}

	/// <summary>
	/// Verändert den Wert (Ladungsstand) der Batterie Leiste auf den übergebenen Wert 
	/// </summary>
	/// <param name="newBatteryLevel">New battery level. (Zwischen 0 und 100)</param>
	public void SetBatteryLevel(float newBatteryLevel) {
		// Set the battery bar's value to the current battery.
		batterySlider.value = currentBattery;
	}

	/// <summary>
	/// Liefert den Prozentwert (0 - 100) der Batterie zurück
	/// </summary>
	/// <returns>The batterie leven.</returns>
	public float GetBatteryLevel() {
		return currentBattery;
	}

	/// <summary>
	/// Gibt true zurück, falls BatteryController gerade im Leerlauf (Kein veränderung der Ladung)
	/// </summary>
	/// <returns><c>true</c> if this instance is idle state; otherwise, <c>false</c>.</returns>
	public bool IsIdleState() {
		return curState == BatteryControllerState.IDLE;
	}
	
	/// <summary>
	/// Läd die Batterie mit linearen Schritten auf
	/// </summary>
	/// <param name="loadPerSec">Prozent, mit der sich Batterie pro Sekunde aufläd (0 - 100).</param>
	public void ChargeBattery(float loadPerSec) {
		batChangePerSec = loadPerSec;
		curState = BatteryControllerState.CHARGE;
	}

	/// <summary>
	/// Entläd die Batterie mit linearen Schritten auf
	/// </summary>
	/// <param name="loadPerSec">Prozent, mit der sich Batterie pro Sekunde entläd (0 - 100).</param>
	public void DischargeBattery(float loadPerSec) {
		batChangePerSec = loadPerSec;
		curState = BatteryControllerState.DISCHARGE;
	}
	
	// Wird jeden Frame aussgeführt
	void Update () 
	{
		switch(curState) {

		// Tue nichts
		case BatteryControllerState.IDLE:					
			break;

		// Lade Batterie auf
		case BatteryControllerState.CHARGE:						
			if(currentBattery <= 100) {
				currentBattery += batChangePerSec * Time.deltaTime;
			}else{
				currentBattery = 100;
				curState = BatteryControllerState.IDLE;
			}
			SetBatteryLevel (currentBattery);
			break;

		// Entlade Batterie
		case BatteryControllerState.DISCHARGE:					
			if(currentBattery >= 0) {
				currentBattery -= batChangePerSec * Time.deltaTime;
			}else{
				currentBattery = 0;
				curState = BatteryControllerState.IDLE;
			}
			SetBatteryLevel (currentBattery);
			break;
		}

	}

	// Enum stellt die Zustände dar, die der BatteryController einnehmen kann
	enum BatteryControllerState{
		IDLE,
		CHARGE,
		DISCHARGE
	}

}
