﻿// ---------------------
//   Author:  O.Feucht 
// ---------------------

using UnityEngine;
using System.Collections;

/// <summary>
/// GameEvent, dass die Batterie langsam leergehen lässt
/// </summary>
public class BatteryLow_EventLogic : GameEvent {

	// Geschwindigkeit, mit der sich die Batterie pro Sekunde entläd
	public float batDecPerSec = 12.5f;

	// Geschwindigkeit, mit der sich die Batterie nach dem erfolgreichen Minispiel pro Sekunde wieder aufläd
	public float batIncPerSec = 40f;

	// Alarmsound, der abgespielt wird
	public AudioClip alarmSound;

	// Zustandsautomat für GameEvent
	private StateMachine stateMachine = new StateMachine ();

	// Steuert die Veränderung der batterie im CarUI
	public BatteryController batController;

	// Das Template UI Panel, das eine Warnung zeigt, wenn die Batterie fast leer ist
	public GameObject TEMPLATE_warningBatteryUI;

	// DIe erzeugte Instanz von "Template_warningBatteryUI" 
	private GameObject warningBatteryUI;

	#region implemented abstract members of GameEvent

	/// <summary>
	/// Einschubmethode für konkretes Event
	/// Wird aufgerufen um den Pre Teil des GameEvents zu initialisieren und zu starten
	/// Mit "Pre Teil des GameEvent" ist der Teil des GameEvents gemeint, der vor dem Minispiel ausgeführt wird
	/// </summary>
	protected override void StartPreEvent ()
	{
		batController.DischargeBattery (batDecPerSec);
		stateMachine.SetState (DischargeBatteryLevelState);
	}

	/// <summary>
	/// Einschubmethode für konkretes Event
	/// Wird aufgerufen um den Post Teil des GameEvents zu initialisieren und zu starten
	/// Mit "Post Teil des GameEvent" ist der Teil des GameEvents gemeint, nach vor dem Minispiel ausgeführt wird
	/// </summary>
	protected override void StartPostEvent ()
	{
		batController.ChargeBattery (batIncPerSec);
		stateMachine.SetState (ChargeBatteryLevelState);
	}

	/// <summary>
	/// Einschubmethode für konkretes Event
	/// Wird in jedem Frame einmal aufgerufen (Loop für Event)
	/// </summary>
	protected override void updateEvent ()
	{
		stateMachine.runState ();
	}

	/// <summary>
	/// Muss einen einzigartigen Namen für jedes GameEvent zurückliefern
	/// </summary>
	/// <returns>The unique name.</returns>
	public override string GetUniqueName ()
	{
		return "BatteryLow";
	}
	 
	
	#endregion

	/// <summary>
	/// Zustand, in dem die Batterie des AutoUIs verringert wird
	/// </summary>
	private void DischargeBatteryLevelState() {
		// Warte bis Batterie fast vollständig entladen
		if (batController.GetBatteryLevel() <= 15) {

			// Zeige Battie fast leer Meldung
			warningBatteryUI = Instantiate(TEMPLATE_warningBatteryUI);
			GameObject canvas = GameObject.FindGameObjectWithTag("Canvas");
			warningBatteryUI.transform.SetParent(canvas.transform, false);

			// Entlade Batterie nun langsamer
			batController.DischargeBattery(4f);

			// Starte Alarmsound
			audioController.StartNewClip (alarmSound, false);

			// Zustandswechsel
			stateMachine.SetState(ShowWarningState);
		}
	}

	/// <summary>
	/// Zustand der aktiv ist, wenn Batterie fast leer
	/// Batterie wird langsamer entladen und eine Warnung wird im UI ausgegeben
	/// </summary>
	private void ShowWarningState() {

		// Warte bis Batterie vollständig entladen
		if (batController.IsIdleState() == true) {
			// Beende Alarmsound
			audioController.StoppMusik();

			// Zerstöre Warning UI
			DestroyImmediate(warningBatteryUI);
			GameEventOver();
		}
	}

	/// <summary>
	/// Zustand, in dem die Batterie des AutoUIs aufgeladen wird
	/// </summary>
	private void ChargeBatteryLevelState() {
		// Warte bis Batterie vollständig entladen (oder aufgeladen)
		if (batController.IsIdleState () == true) {
			GameEventOver();
		}
	}

}
