﻿// ---------------------
//   Author:  O.Feucht 
// ---------------------

using UnityEngine;
using UnityEngine.UI;
using System.Collections;

/// <summary>
/// Kontrolliert den Spieleablauf hinsichtlich der Abfolge und Aufruf von GameEvents und den Minigames
/// </summary>
public class GameStateController : MonoBehaviour {

	// Steuert den kompletten Ablauf von GameEvents (z.B.Hack oder BatteryLow)
	public GameEventController gameEventController;

	// Steuert den kompletten Ablauf von Minispielen
	public MinigameController minigameController;

	// Steuert das Fahrverhalten des Autos
	public DrivingController drivingController;

	// Regelt den Gesammtscore des Spiels
	public ScoreController scoreController;

	// Regelt den Szenenwechsel nach dem GameOver
	public MenueStateController menueStateController;

	// Steuert Hintergrundgeräusche (z.B. Fahrtgeräusche des Autos)
	public AudioSource backgoundSoundController;

	// Das Panel, das bei Gameover gesetzt wird
	public GameObject gameOverUIPrefab;

	/**
	 * Variablen, um GameStateController zu konfigurieren
	 **/

	// Gibt an zu wieviel Prozent ein Event auftritt (Pro Intervall) 
	public float chanceForEvent = 0.5f;

	// Zeitintervall, nachdem ein Event auftretten kann
	public float timeInterval = 6f;

	/// <summary>
	/// Gibt an ob GameEvents auftreten dürfen
	/// Falls Wert false: Es treten keine Events auf und demnach auch keine Minispiele
	/// </summary>
	public bool allowGameEvents = true;
	
	// Gibt an, ob es nach dem Verlieren eines Minispiel ein Hartes GameOver gibt
	// -> Wenn "true" wird Spiel beendet, fall Spiler ein Minispiel verliert
	public bool hardGameOver = true;

	/**
	 * Interne Hilfsvariablen
	 */ 

	// regelt Zustände, die EventController annehmen kann
	private StateMachine stateMachine;

	// Interner Timer zur Zeitmessung
	private float goalTime;

	// Name des letzten aufgetretenen GameEvents
	private string lastEventName;

	// Use this for initialization
	void Awake () {
		drivingController.StopDrinving ();
		stateMachine = new StateMachine ();
		scoreController.Score = 0;
		Random.seed = (int)System.DateTime.Now.Ticks;

		goalTime = Time.time + timeInterval;
		backgoundSoundController.Play ();
		drivingController.StartDriving();
		stateMachine.SetState(WaitForEventState);
	}
	
	// Update is called once per frame
	void Update () {
		stateMachine.runState();
	}

	/// <summary>
	/// Zustand, wenn Auto normal fährt und kein Event oder Minigame läuft.
	/// Aktiviert per Zufall Events
	/// </summary>
	private void WaitForEventState() {
		// Arbeite nur wenn Events aktiviert
		if (allowGameEvents == true) {
			// Überprüfe ob Zeitintervall abgelaufen
			if(Time.time >= goalTime) {
				// Starte per Zufall ein Event
				if(Random.value <= chanceForEvent) {
					StartPreGameEvent();
				}else{
					goalTime = Time.time + timeInterval;
				}
			}
		}
	}
	
	/// <summary>
	/// Startet ein zufälliges Event
	/// </summary>
	private void StartPreGameEvent() {
		lastEventName = gameEventController.StartRandGameEvent ();
		stateMachine.SetState (PreGameEventRunning);
	}

	/// <summary>
	/// Zustand, der aktiv ist, wenn die Pre Phase eines GameEvent läuft (Vor Minispiel)
	/// </summary>
	private void PreGameEventRunning() {
		// Warte bis GameEvent vorbei
		if (gameEventController.GameEventRunning == false) {
			StartMinigame ();
		}
	}

	/// <summary>
	/// Starte ein Minigame
	/// </summary>
	private void StartMinigame() {
		// Starte ein zufälliges Minispiel
		minigameController.StartRandMinigame (lastEventName);
		stateMachine.SetState (MinigameRunning);

		// Halte Auto an
		drivingController.StopDrinving ();

		// Beende normale Hintergrundgeräusche
		backgoundSoundController.Stop ();
	}

	/// <summary>
	/// Zustand, der aktiv ist, wenn Minigame läuft
	/// </summary>
	private void MinigameRunning() {
		// Warte bis Minispiel vorbei ...
		if (minigameController.MinigameRunning == false) {

			// Zähle im Minispiel bekomme Punkt (Score) zum Gesammtscore dazu
			scoreController.AddScore(minigameController.GetScore());

			// Wenn Minispiel verloren -> Ziehe Punkte vom Score ab oder beende Spiel (hardGameOver ?)
			if (minigameController.PlayerWin == false) {
				// Je nach Einstellung wird Spiel beendet oder nur Score abgezogen
				if(hardGameOver == true) {
					StartGameOver();
				}else{
					scoreController.AddScore(-150, 0.4f);
					// Starte den Teil des Events, der nach dem Minispiel ausgeführt wird
					StartPostGameEvent();
				}
			// Wenn Minispiel verloren
			}else{
				// Starte den Teil des Events, der nach dem Minispiel ausgeführt wird
				StartPostGameEvent();
			}


		}
	}

	/// <summary>
	/// Startet den Post Teil des GameEvents, welches das Minispiel ausgelöst hat
	/// </summary>
	private void StartPostGameEvent() {
		gameEventController.StartPostGameEventAction ();
		stateMachine.SetState (PostGameEventRunning);
	}

	/// <summary>
	/// Zustand, der aktiv ist, wenn die Post Phase eines GameEvent läuft (Nach Minispiel)
	/// </summary>
	private void PostGameEventRunning() {
		// Warte bis (Post-Teil) des GameEvents vorbei
		if (gameEventController.GameEventRunning == false) {

			// Lasse Auto weiterfahren
			drivingController.StartDriving();

			// Starte wieder die normalen Hintergrundgeräusche
			backgoundSoundController.Play();

			// Starte wieder von Vorne (Warten auf Start eines GameEvents)
			goalTime = Time.time + timeInterval;
			stateMachine.SetState (WaitForEventState);

		}
	}
			
	/// <summary>
	/// Leitet das Beenden des Spiels ein.
	/// </summary>
	public void StartGameOver() {
		// Stoppe Fahrt
		drivingController.StopDrinving ();

		stateMachine.SetState (WaitForGameOver);
	}

	/// <summary>
	/// Wartet, bis Spiel beendet werden kann
	/// </summary>
	private void WaitForGameOver() {
		// GameOver kann erst gestartet werden, wenn der Score komplett eingeflogen ist
		if (scoreController.IsElementMoving() == false) {

			// Blende UI vom laufenden Events aus
			Transform canvas = GameObject.FindGameObjectWithTag ("Canvas").transform;
			for (int i = 0; i < canvas.childCount; i++) {
				GameObject childPanel = canvas.GetChild(i).gameObject;
				if(!childPanel.tag.Equals("GameUI")) {
					childPanel.SetActive(false);
				}
			}
			// Beende laufende GameEvents und Minispiele
			minigameController.gameObject.SetActive(false);
			gameEventController.gameObject.SetActive(false);

			// Beende fahrtgeräusche
			backgoundSoundController.Stop();

			// Speichere erreichten Score ab
			int score = (int)scoreController.Score;
			ScoreManager scoreManager = gameObject.AddComponent<ScoreManager> ();
			scoreManager.SetCurScore(score);
			scoreManager.StoreScore (scoreManager.GetCurName (), (int)scoreController.Score);

			// Blende gameOver Panel ein
			GameObject go = Instantiate (gameOverUIPrefab);
			go.transform.SetParent (canvas, false);
			
			// Ab jetzt nurnoch Leerlauf
			stateMachine.SetState(WaitForPlayerSubmit);
		}
	}

	/// <summary>
	/// Warte, bis Spieler bestätigt
	/// </summary>
	private void WaitForPlayerSubmit() {
		// Warte auf Bestätigung des Spielers
		if (Input.GetButton ("Submit") && Input.anyKeyDown) {
			menueStateController.setNextScene("Scoreboard");
		}
	}



}
