// ---------------------
//   Author:  O.Feucht 
// ---------------------

using UnityEngine;
using System.Collections;

/// <summary>
/// Vaterklasse für alle Minispiele
/// Regelt allg. Verhalten eines Minispiels (z.B. zuerst Tutorial, dann laufendes Spiel und zum Schluss anzeigen des Highscores)
/// </summary>
public abstract class Minigame : MonoBehaviour {

	// Liste mit dem Namen aller GameEvents, für die dieses Minispiel kompatibel ist
	public string[] compatibleGameEvents;

	// Variable auf die Steuereinheit dieses Minispiels
	private MinigameController minigameController;

	// regelt optisch und logisch den Score des Minispiels verwaltet
	private ScoreController scoreController;

	// Regelt die Ausführung des Tutorials und enthält die anzuzeigenden Panels
	private TutorialController tutController;

	// Regelt den grundlegenden Ablauf eines Minispiels (Tutorial, Ingame, Highscore setzten)
	private StateMachine stateMachine = new StateMachine();

	// Hilfsvaiable zum Temporären merken wer gewonnen hat
	private bool playerWin;

	/// <summary>
	/// Liefert die Steuereinheit für den localen Score zurück
	/// </summary>
	/// <value>The score controller.</value>
	public ScoreController ScoreController{
		get{
			return scoreController;
		}
	}

	/// <summary>
	/// Überprüft ab das GameEvent zu dem Minispiel kompatibel ist
	/// </summary>
	/// <returns><c>true</c> if this instance is compatible to game event the specified gameEventName; otherwise, <c>false</c>.</returns>
	/// <param name="gameEventName">Game event name.</param>
	public bool IsCompatibleToGameEvent(string gameEventName) {
		// Falls Liste leer -> Minispiel ist zu allen GameEvents Kompatibel
		if (compatibleGameEvents.Length == 0) {
			return true;
		}
		// ANsonsten durchsuche die Liste nach übereinstimmung
		foreach (string gEvt in compatibleGameEvents) {
			if(gameEventName.Equals(gEvt)) {
				return true;
			}
		}
		// Keine Übereinstimmung gefunden -> Nicht kompatibel
		return false;
	}

	/// <summary>
	/// Startet das Minispiel
	/// </summary>
	public void StartMinigame(MinigameController controller, ScoreController scoreContoller, TutorialController tutController = null) {
		minigameController = controller;
		this.scoreController = scoreContoller;
		this.tutController = tutController;
		scoreContoller.Score = 0;

		// Starte zuerst das Tutorial falls verfügbar
		if (tutController != null) {
			tutController.StartTutorial ();
			stateMachine.SetState (WaitForTutorialEnding);
		} else {
			InitGame();
			stateMachine.SetState (updateMinigame);
		}
	}
	
	// Zustand, in dem das Tutorial des Minispiels angezeigt wird
	public void WaitForTutorialEnding() {
		// Warte bis Tutorial vorbei und Starte dann eigentliches Spiel (Spiele Logik)
		if (tutController.IsTutorialRunning() == false) {
			InitGame();
			stateMachine.SetState (updateMinigame);
		}
	}

	// Haltet das Minispiel an / Beendet es
	public void GameOver(bool playerWin) {
		// Stoppe Logik des Minispiels
		stateMachine.setIdleState();
		// Leite Ende des Minispiels ein
		stateMachine.SetState (WaitForMinigameEnding);
		this.playerWin = playerWin;
	}
	
	// Update is called once per frame
	// Delegiert an aktuellen Spielezustand
	void Update () {
		stateMachine.runState();
	}

	/// <summary>
	/// In diesem Zustand wird auf das Ende des Minispiels gewartet (Bis Score komplett)
	/// </summary>
	private void WaitForMinigameEnding() {
		
		// Warte bis Score vollständig Zusammengezählt
		if (scoreController.IsElementMoving() == false) {
			// Melde Sieg oder Niederlage an MinigameController
			minigameController.GameOver(playerWin);
		}

	}

	/// <summary>
	/// Einschubmethode für konkretes Minispiel
	/// Wird in jedem Frame einmal aufgerufen (Game-Loop für Minispiel)
	/// </summary>
	protected abstract void updateMinigame ();

	/// <summary>
	/// Einschubmethode für konkretes Minispiel
	/// Wird beim Starten eines jeden Minispiels aufgerufen um alles zu initialisieren
	/// </summary>
	protected abstract void InitGame ();

}
