// ---------------------
//   Author:  O.Feucht 
// ---------------------

using UnityEngine;
using UnityEngine.UI;
using System.Collections;
using System.Collections.Generic;

/// <summary>
/// Verwaltet starten und beenden des Minigames
/// Bindeglied zwischen Klasse "Minigame" und "GameStateController"
/// Alle "Minigame" GameObjects müssen direkte Kinder dieses Objekts sein
/// </summary>
public class MinigameController : MonoBehaviour {

	// Gibt an ob ein Minispiel gerade läuft
	private bool minigameRunning = false;

	// Gibt an ob Spieler gewonnen hat (ACHTUNG: Nur AUssagekräftig wenn Minispiel zu Ende
	private bool playerWin = false;

	// Liste aller GameObjecte, die Minispiele darstellen
	private List<GameObject> minigameList;

	// Referenz auf GameObejct, das Minigame darstellt
	// Muss die Scripte "UI_Connector" und "Minigame" als Komponenten haben (Optional "Tutorial Controller")
	private GameObject curMinigame;

	// Der ScoreController des aktuell laufenden Minispiels
	private ScoreController minigameScoreController;

	// Steuert die Hintergrundmusik
	public MusicController musicController;

	// Das Template für das Score Panel, das in jedes MinigameUI Panel gepackt wird
	// Muss ein Script "MinigameScoreController" beinhalten
	public GameObject scorePanelTemplate;

	/// <summary>
	/// Gibt einen Wert zurück, der angibt ob das Minispiel noch läuft (Nicht fetig ist) <see cref="CreateMiniGame"/> minigame running.
	/// </summary>
	/// <value><c>true</c> Wenn Minigame gerade läuft; otherwise, <c>false</c>.</value>
	public bool MinigameRunning
	{
		get{
			return minigameRunning;
		}
	}

	/// <summary>
	/// Gibt "true" zurück, falls Spieler das Minispiel gewonnen hat <see cref="MinigameController"/> player window.
	/// </summary>
	/// <value><c>true</c> if player window; otherwise, <c>false</c>.</value>
	public bool PlayerWin
	{
		get{
			return playerWin;
		}
	}

	/// <summary>
	/// Gibt den Score für das psezifische Minispiel zurück
	/// </summary>
	/// <value>The score controller.</value>
	public float GetScore() {
		return minigameScoreController.Score;
	}

	/// <summary>
	/// Initalisiert das Script
	/// </summary>
	void Start() {
		// Initalisiere Zufallsgenarator
		Random.seed = System.DateTime.Now.GetHashCode();

		// Nehme alle aktiven Unterobjekte als Minigames in eine Liste aus
		minigameList = new List<GameObject> (10);
		for (int i = 0; i < transform.childCount; i++) {
			GameObject mGame = transform.GetChild(i).gameObject;
			if(mGame.activeSelf == true) {
				minigameList.Add(mGame);
				mGame.SetActive(false);
			}
		}


	}

	/// <summary>
	/// Startet das Minispiel, indem Minspiel-UI  erzeugt wird und Spiele Logik gestartet
	/// </summary>
	public void StartRandMinigame(string gameEventName = "") {

		List<GameObject> aviableMinigameList = minigameList;

		// Falls ein Name eines GameEvents übergeben wurde -> Suche kompatible raus und verwende nur diese
		if (gameEventName != "") {
			List<GameObject> compatibleMinigames = new List<GameObject>();

			// Suche unter allen Minispielen die kompatiblen raus
			foreach(GameObject mGame in minigameList) {
				Minigame mGameScript = mGame.GetComponent<Minigame>();
				if(mGameScript.IsCompatibleToGameEvent(gameEventName)) {
					compatibleMinigames.Add(mGame);
				}
			}
			aviableMinigameList = compatibleMinigames;
		}


		// Starte Musik
		musicController.StartRandomClip ();

		// Setze Zustandsvariable
		minigameRunning = true;

		// Aktiviere Minispiel durch aktiv setzten eines GameObjekts, welches das Minispiel darstellt
		int index = Mathf.RoundToInt(Random.Range (0, aviableMinigameList.Count)) % aviableMinigameList.Count;
		curMinigame = aviableMinigameList [index];
		curMinigame.SetActive(true);

		// Erzeuge Minigame UI Panel
		GameObject minigameUI = curMinigame.GetComponent<UIConnector> ().CreateMinigameUI ();

		// Erzeuge Score Panel
		GameObject scorePanel = Instantiate (scorePanelTemplate);
		minigameScoreController = scorePanel.GetComponent<ScoreController> ();
		scorePanel.transform.SetParent (minigameUI.transform, false);

		// Starte Minigame
		TutorialController tutController = curMinigame.GetComponent<TutorialController> ();
		curMinigame.GetComponent<Minigame>().StartMinigame(this, minigameScoreController, tutController);
	}

	/// <summary>
	/// Minispiel ist vorbei und wird nun Beendet
	/// </summary>
	/// <param name="win">Wenn <c>true</c> ist das Minispiel gewonnen worden</param>
	public void GameOver(bool isWin = true) {

		// Beende Musik
		musicController.StoppMusik ();

		// Setze Zustandsvariablen
		minigameRunning = false;
		playerWin = isWin;

		// Zerstöre Minigame UI Panel
		curMinigame.GetComponent<UIConnector>().DestroyMinigameUI();

		// Deaktiviere GameObject, das Minispiel Scripte ("Minigame", "UICcnnector")
		curMinigame.SetActive (false);
	}

}
