﻿// ---------------------
//   Author:  O.Feucht 
// ---------------------

using UnityEngine;
using System.Text;
using System.Collections;

public class RememberText_GameLogic : Minigame {

	/*
	 * Variablen, um Spiel zu konfigurieren
	 */

	// Miimale Zeitdauer, in der zu merkender Text angezeigt wird
	public float minShowTextTime = 6f;

	// Wird zur Mindestzeigdauer (minShowTextTime) dazugezählt um auch bei langen zu merkenden Texten eine Chance zu haben
	public float showTextTimePerChar = 1f;

	// Zeitdauer, in der Spieler den gemerkten Text eingeben kann
	public float playerInputTime = 15f;

	// Gibt an, wieviele Runden es gibt und wieviele Zeichen man sich pro Runde merken muss
	public int[] textLenghPerRound = {3, 4, 5, 6, 7, 8, 9};

	// Punktzahl, die man pro gewonnene Runde für jedes Zeichen des zumerkenden Textes bekommt
	public float scorePerChar = 5;

	// Multiplikator, mit der Gesammtscore des Minigames nach jeder gewonnenen Runde multipliziert wird
	public float scoreMultiPerRound = 1.25f;

	// Multiplikator, mit der Gesammtscore des Minigames nach der letzten Runde multipliziert wird
	public float scorePerWinLastRound = 2f;

	/*
	 * Hilfsvariablen und andere Referenzen
	 */ 

	// Zustandsautomat, um den Ablauf des Spiels zu steuern
	private StateMachine stateMachine;

	// Regelt verbindung zum UI
	private RememberText_UIConnector uiConnector;

	// Hilfsvariable zur Zeitmessung
	private float time1;

	// Index der Spielphase
	private int roundIndex;

	// Text den sich Spieler merken muss
	private string remindText;


	#region implemented abstract members of Minigame
	/// <summary>
	/// Einschubmethode für konkretes Minispiel
	/// Wird in jedem Frame einmal aufgerufen (Game-Loop für Minispiel)
	/// </summary>
	protected override void updateMinigame ()
	{
		stateMachine.runState();
	}

	/// <summary>
	/// Einschubmethode für konkretes Minispiel
	/// Wird beim Starten eines jeden Minispiels aufgerufen um alles zu initialisieren
	/// </summary>
	protected override void InitGame ()
	{
		roundIndex = 0;
		Random.seed = System.DateTime.Now.GetHashCode();
		stateMachine = new StateMachine ();
		uiConnector = GetComponent<RememberText_UIConnector> ();
		uiConnector.Init();
		StartShowingText ();
	}
	#endregion

	/// <summary>
	/// Leitet die Spielphase "ShowTextState" ein, in der der Spieler den zu merkenden Text angezeigt bekommt
	/// </summary>
	private void StartShowingText() {
		time1 = Time.time + minShowTextTime + showTextTimePerChar * textLenghPerRound[roundIndex];
		remindText = GetRndText ();
		uiConnector.SetReadState (remindText);
		stateMachine.SetState(ShowTextState);
	}
	
	/// <summary>
	/// Spielezustand, bei dem der zu merkende Text angezeigt wird.
	/// </summary>
	private void ShowTextState() {
		// Aktualisiere runterzählende zeit
		uiConnector.ShowTime (time1 - Time.time);
		// Warte bis Zeit zum anzeigen vorbei ODER Spieler bestätigt hat (Enter, Backspace, ...)
		if ( (Time.time >= time1) || (uiConnector.HasPlayerSubmitted() == true) ) {
			stateMachine.SetState(WaitUserImputState);
			time1 = Time.time + playerInputTime;
			uiConnector.SetInputState();
		}
	}

	private void WaitUserImputState() {
		// Aktualisiere runterzählende zeit
		uiConnector.ShowTime (time1 - Time.time);

		// Falls Spieler Eingabe Bestätigt hat, überprüfe Inhalt des InputFields auf richtigkeit
		if (uiConnector.HasPlayerSubmitted() == true) {
			PlayerRoundOver();
			return;
		}

		// Falls Zeit abgelaufen und Spiler noch nicht bestätigt hat überprüfe Inhalt von InputField auf Richtigkeit
		if (Time.time >= time1) {
			PlayerRoundOver();
			return;
		}
	}

	/// <summary>
	/// Wird ausgeführt wenn die aktuelle Runde vorbei ist
	/// Überprüft ob Spieler gewonnen hat, das Spiel weiter geht oder ob Spieler verloren hat.
	/// </summary>
	private void PlayerRoundOver() {
		// Richtige Eingabe des Spielers
		if (remindText.Equals (uiConnector.GetUserInputText ())) {
			// Erhöhe Score um das x-fache, der gemerken Zahlen und multipliziere Score dann mit Multiplikator
			ScoreController.AddScore(textLenghPerRound[roundIndex] * scorePerChar);
			ScoreController.MultiplyScore(scoreMultiPerRound, 0.25f);

			// Überprüfe ob nächste Runde gestartet werden muss oder ob bereits letzte Runde
			if(roundIndex < textLenghPerRound.Length - 1) {
				// Starte neue Runde des Minispiels
				roundIndex++;
				StartShowingText();
			}else{
				// Alle Runden des Minispiels geschafft - Minispiel erfolgreich vorbei
				ScoreController.MultiplyScore(scorePerWinLastRound, 0.3f);
				GameOver(true);
			}
		}
		// Keine oder falsche Eingabe des Spielers
		else {
			// Falls Spieler mindestens die Hälfte der Runden geschafft hat ODER sich mehr als 6 Zeichen merken musste -> Minispiel gewonnen
			if( (roundIndex >= textLenghPerRound.Length / 2) || (textLenghPerRound[roundIndex] > 6) ) {
				GameOver(true);
			}
			// Ansonsten Beende Spiel als Niederlage
			else{
				GameOver(false);
			}
		}
	}

	private string GetRndText() {
		StringBuilder builder = new StringBuilder ();
		for (int i = 0; i < textLenghPerRound[roundIndex]; i++) {
			builder.Append(Random.Range (0, 10));
		}
		return builder.ToString ();
	}
}
