﻿// ---------------------
//   Author:  O.Feucht 
// ---------------------

using UnityEngine;
using UnityEngine.UI;
using System.Collections;

using System;

/// <summary>
/// Ermöglicht die Bewegung eines beliebigen GameObjects mit linearer Geschwindigkeit von einem Start- zu einem Endpunkt 
/// </summary>
public class ElementMover {

	// Endposition, auf die sich Element zu bewegt
	protected Vector3 endPos;

	// Referenz auf Element, das bewegt wird
	protected RectTransform movedElement;

	// Geschwindigkeit, mit der sich Element bewegt
	protected float movingSpeed;

	// Richtung, in die sich Element bewegt
	protected Vector3 movingDir;

	// Hilfsvariablen, die für das ABruchkriterium der Bewegung gebraucht werden
	bool isSmalerX, isSmalerY, isSmalerZ;

	/// <summary>
	/// Konstruktor für MoveElement<see cref="MoveElement"/> class.
	/// </summary>
	/// <param name="transform">Element, das von der aktuellen Position bis zu einer Endposition bewegt werden soll</param>
	/// <param name="endPos">End position.</param>
	/// <param name="modingSpeed">Moding speed.</param>
	public ElementMover(RectTransform movedElement, Vector3 endPos, float movingSpeed) {
		this.movedElement = movedElement;
		this.endPos = endPos;
		this.movingSpeed = movingSpeed;
		movingDir = endPos - movedElement.position;
		
		// Lege Abbruchkriterium fest
		Vector3 curPos = movedElement.position;
		isSmalerX = endPos.x < curPos.x;
		isSmalerY = endPos.y < curPos.y;
	}

	/// <summary>
	/// Das Element wird bewegt, falls dieses noch nicht am Ziel ist.
	/// Rückgabewert gibt an, ob Element schon das Bewegungsziel erreicht hat 
	/// Return false == Element hat Ziel erreicht, daher keine Bewegung mehr 
	/// </summary>
	/// <returns><c>true</c>, Wenn das Element bewegt wurde, <c>false</c> Wenn das Element bereits am Ziel ist.</returns>
	public virtual bool MoveElement() {
		Vector3 curPos = movedElement.position;

		// Überprüfe auf Abbruchkriterium der Bewegung
		if (isSmalerX != endPos.x < curPos.x || isSmalerY != endPos.y < curPos.y || isSmalerZ != endPos.z < curPos.z) {
			return false;
		}
		// Falls kein Abbruch -> Bewege Objekt

		// Berechne Bewegung zum Ziel
		float deltaX = movingDir.x * Time.deltaTime * movingSpeed;
		float deltaY = movingDir.y * Time.deltaTime * movingSpeed;

		// Bewege zum Ziel
		curPos.x += deltaX;
		curPos.y += deltaY;

		//Setze bewegendes Element auf neue Position
		movedElement.position = curPos;

		return true;
	}

	/// <summary>
	/// Gibt eine Referenz auf das sich Bewegende GameObject zurück
	/// </summary>
	/// <returns>The moving game object.</returns>
	public GameObject GetMovingGameObject() {
		return movedElement.gameObject;
	}



}
