﻿// ---------------------
//   Author:  O.Feucht 
// ---------------------

using UnityEngine;
using UnityEngine.UI;
using System.Collections;
using System.Collections.Generic;

/// <summary>
/// Steuert die optische Anzeige des Scores in einem Minispiel
/// </summary>
public class ScoreController : MonoBehaviour {

	// Das Textfeld, in dem der aktuelle Score angezeigt wird
	public Text currentScoreText;

	// Start und Endposition, für des einfliegen des neuen Scores
	public RectTransform startPos;
	public RectTransform endPos;

	// Die Geschwindigkeit, mit der der Text einfliegt
	// (Wert zwischen 0 und 1)
	public float slidingSpeed = 1f;

	// Template für den Text, der einfliegt
	public GameObject slidingTextTemplate;

	// Der aktuell angezeigt score
	private float score;

	// Liste mit allen Score Texten, die sich gerade bewegen
	private List<ScoreMover> movingElements = new List<ScoreMover> (10);

 	/// <summary>
	/// Liefert den aktuellen Score zurück oder
	/// Setzt den übergebenen Score als neuen Gesammtscore und zeigt diesen an
 	/// </summary>
 	/// <value>The score.</value>
	public float Score{
		get{
			return score;
		}
		set{
			currentScoreText.text = string.Format("{0:0.##}", value);
			score = value;
		}
	}
	

	/// <summary>
	/// Addiert einen Betrag zum Gesammtscore des Minispiels
	/// </summary>
	/// <param name="amount">Amount.</param>
	public void AddScore(float amount, float waitTime = 0) {
		// Wandle Betrag in passenden String um
		string shownText = "";
		if (amount > 0) {
			shownText += "+";
		}
		shownText += string.Format("{0:0.##}", amount);
		// Erstelle passendes Delegate
		AddMovingScore(
			delegate (float currentScore) {
				return (currentScore + amount);
			},
			shownText,
		waitTime
		);
	}

	/// <summary>
	/// Multipliziert den Gesammtscore des Minispiels mit einem Betrag
	/// </summary>
	/// <param name="scale">Scale.</param>
	public void MultiplyScore(float scale, float waitTime = 0) {
		// Wandle Betrag in passenden String um
		string shownText = "*" + string.Format("{0:0.##}", scale);
		// Erstelle passendes Delegate
		AddMovingScore(
			delegate (float currentScore) {
			return (currentScore * scale);
		},
		shownText,
		waitTime
		);
	}
	
	/// <summary>
	/// Leitet das einschweben eines neuen Score Elements ein
	/// </summary>
	/// <param name="ScoreEndAction">Funktion, die auf den aktuellen Score des Minispiels angewand wird, wenn das Element am Zielpunkt angekommen</param>
	/// <param name="shownText">Der angeziegte Text, des Bewegenden Score Elements</param>
	private void AddMovingScore(ScoreAction ScoreEndAction, string shownText, float waitTime) {
		// Erzeuge aus dem template ein neues Textfeld
		GameObject slidingText = Instantiate (slidingTextTemplate);
		slidingText.transform.SetParent(slidingTextTemplate.transform.parent, false);

		//GameObject slidingText = slidingTextTemplate;
		slidingText.GetComponent<Text>().text = shownText;

		// Setze das Textfeld an die Startposition
		slidingText.GetComponent<RectTransform>().position = startPos.position;

		// Erzeuge "ScoreMover" oder "DelayedScoreMover", welcher die Bewegung des Textfeldes regelt und hänge dieses an die Liste an
		ScoreMover movEle;
		if (waitTime > 0) {
			movEle = new DelayedScoreMover (slidingText.GetComponent<RectTransform>(), endPos.position, this.slidingSpeed, waitTime);
		} else {
			movEle = new ScoreMover (slidingText.GetComponent<RectTransform>(), endPos.position, this.slidingSpeed);
		}
		movEle.ScoreEndAction = ScoreEndAction;
		movingElements.Add(movEle);
	}

	/// <summary>
	/// Gibt zurück, ob sich Element (Score Texte) bewegen
	/// </summary>
	/// <returns><c>true</c> Wenn es mind 1 sich bewegenden Score Text gibt; otherwise, <c>false</c>.</returns>
	public bool IsElementMoving() {
		return (movingElements.Count != 0);
	}

	// Update is called once per frame
	void Update () {

		// Bewege alle Score Objekte weiter
		for (int i = 0; i < movingElements.Count; i++) {
			ScoreMover movEle = movingElements[i];
			bool canMove = movEle.MoveElement();
			// Falls Score Element am Ziel -> Löschen und Score erhöhen
			if(canMove == false) {
				// Ermittle neuen Gesammtscore im Minispiel
				float newScore = movEle.ScoreEndAction(score);
				Score = newScore;
				// Zerstöre das Bewegte Score Element
				DestroyImmediate(movEle.GetMovingGameObject());
				movingElements.Remove(movEle);
			}
		}

	}
}
